// background.js - 백그라운드 서비스 워커

// 확장프로그램 설치 시
chrome.runtime.onInstalled.addListener(() => {
    console.log('[셀프마케터 태그 분석] 확장프로그램 설치됨');
});

// 메시지 핸들러
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'fetchProductPage') {
        // 상품 페이지 fetch
        fetchProductPage(request.url)
            .then(html => sendResponse({ success: true, html }))
            .catch(error => sendResponse({ success: false, error: error.message }));
        return true; // 비동기 응답
    }
    
    if (request.action === 'extractTagsFromHtml') {
        const tags = extractTagsFromHtml(request.html);
        sendResponse({ success: true, tags });
        return true;
    }
});

// 상품 페이지 HTML 가져오기
async function fetchProductPage(url) {
    const response = await fetch(url, {
        headers: {
            'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
            'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
            'Accept-Language': 'ko-KR,ko;q=0.9,en-US;q=0.8,en;q=0.7'
        }
    });
    
    if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
    }
    
    return await response.text();
}

// HTML에서 태그 추출
function extractTagsFromHtml(html) {
    const tags = [];
    
    // 1. 관련 태그 섹션 찾기
    const patterns = [
        // 스마트스토어 관련 태그
        /관련\s*태그[\s\S]*?<\/(?:div|section|ul)>/gi,
        // 해시태그 직접 매칭
        /#([가-힣a-zA-Z0-9_]{2,20})/g,
        // class에 tag가 포함된 요소
        /class="[^"]*tag[^"]*"[^>]*>([^<]+)</gi
    ];
    
    // 관련 태그 섹션 추출
    const relatedTagMatch = html.match(/관련\s*태그[\s\S]*?(?:<\/div>|<\/section>)/i);
    if (relatedTagMatch) {
        const sectionHtml = relatedTagMatch[0];
        const hashtagMatches = sectionHtml.match(/#([가-힣a-zA-Z0-9_]+)/g);
        if (hashtagMatches) {
            tags.push(...hashtagMatches);
        }
    }
    
    // 전체 HTML에서 해시태그 추출 (관련태그 영역 한정)
    const tagAreaPatterns = [
        /<[^>]*class="[^"]*(?:tag|keyword|hash)[^"]*"[^>]*>([^<]*#[^<]*)<\/[^>]*>/gi,
        /data-nclick="[^"]*tag[^"]*"[^>]*>([^<]+)</gi
    ];
    
    tagAreaPatterns.forEach(pattern => {
        let match;
        while ((match = pattern.exec(html)) !== null) {
            const text = match[1];
            const hashMatches = text.match(/#([가-힣a-zA-Z0-9_]+)/g);
            if (hashMatches) {
                tags.push(...hashMatches);
            }
        }
    });
    
    // 중복 제거 및 정리
    const cleanTags = [...new Set(tags)]
        .map(tag => tag.replace(/^#/, '').trim())
        .filter(tag => tag.length >= 2 && tag.length <= 20);
    
    return cleanTags;
}

// 확장프로그램 아이콘 클릭 시 (네이버쇼핑이 아닌 경우)
chrome.action.onClicked.addListener((tab) => {
    // popup.html이 있으므로 이 이벤트는 발생하지 않지만, 
    // 혹시 모를 경우를 대비
});
