// content.js - 네이버쇼핑 검색 결과 페이지에서 상품 정보 수집

// 메시지 수신
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'getProductLinks') {
        const links = collectProductLinks(request.count || 40);
        sendResponse(links);
    }
    return true; // 비동기 응답을 위해 필요
});

// 상품 링크 수집
function collectProductLinks(count) {
    const productLinks = [];
    
    // 네이버쇼핑 검색 결과 상품 셀렉터들
    const selectors = [
        // 일반 검색 결과
        'div.basicList_item__0T9JD a.basicList_link__JLQJf',
        'div.product_item__MDtDF a.product_link__TrAac',
        // 구버전 셀렉터
        'li.basicList_item a.basicList_link',
        'div.adProduct_item a.adProduct_link',
        // 통합 검색
        '.shopList_product a[href*="smartstore.naver.com"]',
        '.shopList_product a[href*="brand.naver.com"]',
        '.shopList_product a[href*="shopping.naver.com/products"]',
        // 최신 버전 (2024~)
        '[class*="product_item"] a[href*="naver.com"]',
        '[class*="basicList"] a[href*="naver.com"]'
    ];
    
    // 모든 상품 링크 수집
    const allLinks = new Set();
    
    selectors.forEach(selector => {
        try {
            const elements = document.querySelectorAll(selector);
            elements.forEach(el => {
                const href = el.href || el.getAttribute('href');
                if (href && isValidProductLink(href)) {
                    allLinks.add(href);
                }
            });
        } catch (e) {
            console.log('셀렉터 오류:', selector, e);
        }
    });
    
    // 추가: data-nclick 속성으로 찾기
    document.querySelectorAll('a[data-nclick*="product"]').forEach(el => {
        const href = el.href || el.getAttribute('href');
        if (href && isValidProductLink(href)) {
            allLinks.add(href);
        }
    });
    
    // 추가: 상품 이미지 링크에서 부모 a 태그 찾기
    document.querySelectorAll('img[alt]').forEach(img => {
        const parent = img.closest('a');
        if (parent) {
            const href = parent.href || parent.getAttribute('href');
            if (href && isValidProductLink(href)) {
                allLinks.add(href);
            }
        }
    });
    
    // 배열로 변환하고 count만큼 자르기
    const linksArray = Array.from(allLinks);
    
    // 광고 상품 제외 (선택적)
    const filteredLinks = linksArray.filter(link => {
        // 광고는 보통 adcr 파라미터가 있음
        // 필요하면 광고도 포함하려면 이 필터 제거
        return true;
    });
    
    return filteredLinks.slice(0, count);
}

// 유효한 상품 링크인지 확인
function isValidProductLink(url) {
    if (!url) return false;
    
    const validDomains = [
        'smartstore.naver.com',
        'brand.naver.com',
        'shopping.naver.com/products',
        'search.shopping.naver.com/catalog'
    ];
    
    return validDomains.some(domain => url.includes(domain));
}

// 페이지에서 직접 태그 수집 (상세페이지용)
function collectTagsFromPage() {
    const tags = [];
    
    // 관련 태그 영역 찾기
    const tagContainers = document.querySelectorAll([
        '[class*="tag"]',
        '[class*="keyword"]',
        '[data-nclick*="tag"]'
    ].join(','));
    
    tagContainers.forEach(container => {
        const text = container.textContent;
        const hashtagMatches = text.match(/#([가-힣a-zA-Z0-9_]+)/g);
        if (hashtagMatches) {
            tags.push(...hashtagMatches);
        }
    });
    
    // 상품 상세의 관련 태그
    const relatedTagSection = document.querySelector('[class*="relatedTag"], [class*="related_tag"]');
    if (relatedTagSection) {
        const anchors = relatedTagSection.querySelectorAll('a');
        anchors.forEach(a => {
            const tagText = a.textContent.trim();
            if (tagText.startsWith('#')) {
                tags.push(tagText);
            }
        });
    }
    
    return [...new Set(tags)]; // 중복 제거
}

// 현재 페이지 정보 반환
function getPageInfo() {
    const url = new URL(window.location.href);
    return {
        url: window.location.href,
        query: url.searchParams.get('query'),
        isSearchPage: window.location.hostname === 'search.shopping.naver.com',
        productCount: collectProductLinks(100).length
    };
}

// 페이지 로드 완료 시 초기화
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
} else {
    init();
}

function init() {
    console.log('[셀프마케터 태그 분석] 확장프로그램 로드됨');
    
    // 검색 결과 페이지인 경우 상품 수 표시
    if (window.location.hostname === 'search.shopping.naver.com') {
        const count = collectProductLinks(100).length;
        console.log(`[셀프마케터 태그 분석] ${count}개 상품 감지됨`);
    }
}
