// popup.js - 셀프마케터 태그 분석 확장프로그램

const SELFMARKETER_API = 'https://selfmarketer.co.kr';
let collectedTags = [];
let checkResults = [];

// 탭 전환
document.querySelectorAll('.tab').forEach(tab => {
    tab.addEventListener('click', () => {
        document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
        document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
        tab.classList.add('active');
        document.getElementById('tab-' + tab.dataset.tab).classList.add('active');
    });
});

// 태그 분석 시작
document.getElementById('btnAnalyze').addEventListener('click', async () => {
    const btn = document.getElementById('btnAnalyze');
    const progressArea = document.getElementById('progressArea');
    const resultArea = document.getElementById('resultArea');
    const errorMessage = document.getElementById('errorMessage');
    const range = parseInt(document.getElementById('analyzeRange').value);
    
    // 초기화
    btn.disabled = true;
    btn.innerHTML = '<span class="loading-spinner"></span>분석 중...';
    progressArea.classList.add('active');
    resultArea.classList.remove('active');
    errorMessage.style.display = 'none';
    collectedTags = [];
    
    try {
        // 현재 탭이 네이버쇼핑 검색 결과인지 확인
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        
        if (!tab.url.includes('search.shopping.naver.com')) {
            throw new Error('네이버쇼핑 검색 결과 페이지에서 실행해주세요.');
        }
        
        // 검색어 추출
        const urlParams = new URL(tab.url);
        const keyword = urlParams.searchParams.get('query');
        
        if (!keyword) {
            throw new Error('검색어를 찾을 수 없습니다.');
        }
        
        updateProgress(0, `"${keyword}" 상품 목록 수집 중...`);
        
        // content script로 상품 링크 수집 요청
        const productLinks = await chrome.tabs.sendMessage(tab.id, {
            action: 'getProductLinks',
            count: range
        });
        
        if (!productLinks || productLinks.length === 0) {
            throw new Error('상품을 찾을 수 없습니다. 검색 결과 페이지인지 확인해주세요.');
        }
        
        updateProgress(5, `${productLinks.length}개 상품 발견, 태그 수집 시작...`);
        
        // 각 상품 상세페이지에서 태그 수집
        const allTags = [];
        for (let i = 0; i < productLinks.length; i++) {
            const link = productLinks[i];
            const progress = 5 + Math.floor((i / productLinks.length) * 90);
            updateProgress(progress, `${i + 1}/${productLinks.length} 상품 분석 중...`);
            
            try {
                const tags = await fetchProductTags(link);
                if (tags && tags.length > 0) {
                    allTags.push(...tags);
                }
            } catch (e) {
                console.log('태그 수집 실패:', link, e);
            }
            
            // 딜레이 (네이버 차단 방지)
            await delay(300);
        }
        
        updateProgress(95, '태그 집계 중...');
        
        // 태그 빈도 집계
        const tagCount = {};
        allTags.forEach(tag => {
            const cleanTag = tag.replace(/^#/, '').trim();
            if (cleanTag) {
                tagCount[cleanTag] = (tagCount[cleanTag] || 0) + 1;
            }
        });
        
        // 빈도순 정렬
        collectedTags = Object.entries(tagCount)
            .sort((a, b) => b[1] - a[1])
            .map(([tag, count]) => ({ tag, count }));
        
        updateProgress(100, '완료!');
        
        // 결과 표시
        displayTagResults(keyword);
        
    } catch (error) {
        errorMessage.textContent = error.message;
        errorMessage.style.display = 'block';
    } finally {
        btn.disabled = false;
        btn.innerHTML = '🔍 태그 분석 시작';
    }
});

// 상품 상세페이지에서 태그 수집
async function fetchProductTags(productUrl) {
    try {
        const response = await fetch(productUrl, {
            headers: {
                'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
            }
        });
        const html = await response.text();
        
        // "관련 태그" 섹션에서 태그 추출
        const tags = [];
        
        // 스마트스토어/브랜드스토어 태그 패턴
        const patterns = [
            // JSON-LD 데이터에서 추출
            /"keywords":\s*\[([^\]]+)\]/gi,
            // 관련 태그 영역
            /class="[^"]*tag[^"]*"[^>]*>#([^<]+)/gi,
            // 해시태그 패턴
            /#([가-힣a-zA-Z0-9]+)/g
        ];
        
        // 관련태그 섹션 찾기 (네이버쇼핑 상품 상세)
        const tagSectionMatch = html.match(/관련\s*태그[\s\S]*?<\/div>/i);
        if (tagSectionMatch) {
            const hashtagMatches = tagSectionMatch[0].match(/#([가-힣a-zA-Z0-9_]+)/g);
            if (hashtagMatches) {
                tags.push(...hashtagMatches);
            }
        }
        
        // 태그 클래스에서 추출
        const tagClassMatches = html.match(/class="[^"]*tag[^"]*"[^>]*>([^<]+)</gi);
        if (tagClassMatches) {
            tagClassMatches.forEach(match => {
                const tagMatch = match.match(/>([^<]+)</);
                if (tagMatch && tagMatch[1].includes('#')) {
                    tags.push(tagMatch[1]);
                }
            });
        }
        
        // data-nclick 태그 추출 (네이버 고유)
        const nclickMatches = html.match(/data-nclick="[^"]*tag[^"]*"[^>]*>([^<]+)</gi);
        if (nclickMatches) {
            nclickMatches.forEach(match => {
                const tagMatch = match.match(/>([^<]+)</);
                if (tagMatch) {
                    const cleanTag = tagMatch[1].trim();
                    if (cleanTag.startsWith('#') || /^[가-힣a-zA-Z0-9]+$/.test(cleanTag)) {
                        tags.push(cleanTag);
                    }
                }
            });
        }
        
        return [...new Set(tags)]; // 중복 제거
        
    } catch (error) {
        console.error('태그 수집 오류:', error);
        return [];
    }
}

// 결과 표시
function displayTagResults(keyword) {
    const resultArea = document.getElementById('resultArea');
    const tagList = document.getElementById('tagList');
    
    if (collectedTags.length === 0) {
        tagList.innerHTML = '<div style="color:#888;font-size:13px;">수집된 태그가 없습니다.</div>';
    } else {
        tagList.innerHTML = collectedTags.slice(0, 50).map(item => 
            `<span class="tag-item">#${item.tag}<span class="count">${item.count}</span></span>`
        ).join('');
    }
    
    resultArea.classList.add('active');
    
    // 저장 (로컬스토리지)
    chrome.storage.local.set({
        lastAnalysis: {
            keyword: keyword,
            tags: collectedTags,
            timestamp: Date.now()
        }
    });
}

// 셀프마케터로 전송
document.getElementById('btnSendToSelfmarketer').addEventListener('click', async () => {
    const btn = document.getElementById('btnSendToSelfmarketer');
    btn.disabled = true;
    btn.innerHTML = '<span class="loading-spinner"></span>전송 중...';
    
    try {
        const data = await chrome.storage.local.get('lastAnalysis');
        if (!data.lastAnalysis) {
            throw new Error('전송할 데이터가 없습니다.');
        }
        
        // 셀프마케터 API로 전송
        const response = await fetch(`${SELFMARKETER_API}?action=saveHashtagAnalysis`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                action: 'saveHashtagAnalysis',
                type: 'tagAnalysis',
                keyword: data.lastAnalysis.keyword,
                tags: data.lastAnalysis.tags,
                timestamp: data.lastAnalysis.timestamp
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            // 셀프마케터 페이지 열기
            chrome.tabs.create({
                url: `${SELFMARKETER_API}/mobile-hashtag-analysis.html?view=analysis&id=${result.id}`
            });
        } else {
            throw new Error(result.error || '전송 실패');
        }
        
    } catch (error) {
        alert('전송 실패: ' + error.message);
    } finally {
        btn.disabled = false;
        btn.innerHTML = '📤 셀프마케터로 전송';
    }
});

// ========== 내 태그 확인 ==========
document.getElementById('btnCheckTags').addEventListener('click', async () => {
    const btn = document.getElementById('btnCheckTags');
    const progressArea = document.getElementById('checkProgressArea');
    const resultArea = document.getElementById('checkResultArea');
    const errorMessage = document.getElementById('checkErrorMessage');
    
    const mainKeyword = document.getElementById('mainKeyword').value.trim();
    const checkTagsInput = document.getElementById('checkTags').value.trim();
    
    if (!mainKeyword) {
        errorMessage.textContent = '메인 키워드를 입력해주세요.';
        errorMessage.style.display = 'block';
        return;
    }
    
    if (!checkTagsInput) {
        errorMessage.textContent = '확인할 태그를 입력해주세요.';
        errorMessage.style.display = 'block';
        return;
    }
    
    const tagsToCheck = checkTagsInput.split(',').map(t => t.trim().replace(/^#/, '')).filter(t => t);
    
    if (tagsToCheck.length === 0) {
        errorMessage.textContent = '유효한 태그를 입력해주세요.';
        errorMessage.style.display = 'block';
        return;
    }
    
    // 초기화
    btn.disabled = true;
    btn.innerHTML = '<span class="loading-spinner"></span>확인 중...';
    progressArea.classList.add('active');
    resultArea.classList.remove('active');
    errorMessage.style.display = 'none';
    checkResults = [];
    
    try {
        // 1. 메인 키워드 카테고리 확인
        updateCheckProgress(0, `"${mainKeyword}" 카테고리 확인 중...`);
        const mainCategory = await getKeywordCategory(mainKeyword);
        
        if (!mainCategory) {
            throw new Error('메인 키워드의 카테고리를 찾을 수 없습니다.');
        }
        
        updateCheckProgress(20, `기준 카테고리: ${mainCategory}`);
        
        // 2. 각 태그의 카테고리 확인
        for (let i = 0; i < tagsToCheck.length; i++) {
            const tag = tagsToCheck[i];
            const progress = 20 + Math.floor((i / tagsToCheck.length) * 70);
            updateCheckProgress(progress, `"${tag}" 카테고리 확인 중...`);
            
            try {
                const tagCategory = await getKeywordCategory(tag);
                const isMatch = tagCategory && isCategoryMatch(mainCategory, tagCategory);
                
                checkResults.push({
                    tag: tag,
                    category: tagCategory || '찾을 수 없음',
                    mainCategory: mainCategory,
                    match: isMatch
                });
            } catch (e) {
                checkResults.push({
                    tag: tag,
                    category: '오류',
                    mainCategory: mainCategory,
                    match: false
                });
            }
            
            await delay(500);
        }
        
        updateCheckProgress(100, '완료!');
        
        // 결과 표시
        displayCheckResults(mainKeyword, mainCategory);
        
    } catch (error) {
        errorMessage.textContent = error.message;
        errorMessage.style.display = 'block';
    } finally {
        btn.disabled = false;
        btn.innerHTML = '✅ 카테고리 일치 확인';
    }
});

// 키워드로 검색해서 카테고리 추출
async function getKeywordCategory(keyword) {
    try {
        const searchUrl = `https://search.shopping.naver.com/search/all?query=${encodeURIComponent(keyword)}`;
        const response = await fetch(searchUrl);
        const html = await response.text();
        
        // 카테고리 추출 패턴들
        const patterns = [
            // 검색 결과의 카테고리 필터
            /"category[^"]*":\s*"([^"]+)"/i,
            // 카테고리 경로
            /카테고리[^>]*>([^<]+)/i,
            // 브레드크럼
            /"categoryName":\s*"([^"]+)"/i
        ];
        
        for (const pattern of patterns) {
            const match = html.match(pattern);
            if (match && match[1]) {
                return match[1].trim();
            }
        }
        
        // 상위 상품의 카테고리 추출 시도
        const catMatch = html.match(/"catId":\s*"(\d+)".*?"catName":\s*"([^"]+)"/);
        if (catMatch) {
            return catMatch[2];
        }
        
        return null;
        
    } catch (error) {
        console.error('카테고리 조회 오류:', error);
        return null;
    }
}

// 카테고리 일치 확인 (대분류 기준)
function isCategoryMatch(mainCat, tagCat) {
    if (!mainCat || !tagCat) return false;
    
    // 대분류 추출 (첫 번째 > 이전 또는 전체)
    const mainFirst = mainCat.split('>')[0].trim().toLowerCase();
    const tagFirst = tagCat.split('>')[0].trim().toLowerCase();
    
    // 완전 일치 또는 포함 관계
    return mainFirst === tagFirst || 
           mainCat.toLowerCase().includes(tagCat.toLowerCase()) ||
           tagCat.toLowerCase().includes(mainCat.toLowerCase());
}

// 내 태그 확인 결과 표시
function displayCheckResults(mainKeyword, mainCategory) {
    const resultArea = document.getElementById('checkResultArea');
    const resultList = document.getElementById('checkResultList');
    
    let html = `
        <div class="category-result" style="background:#f8f9fa;border-color:#0069ff;">
            <div class="category-tag">📌 기준: ${mainKeyword}</div>
            <div class="category-info">카테고리: ${mainCategory}</div>
        </div>
    `;
    
    checkResults.forEach(item => {
        const matchClass = item.match ? 'match' : 'mismatch';
        const icon = item.match ? '✅' : '❌';
        const statusClass = item.match ? 'success' : 'error';
        const statusText = item.match ? '일치' : '불일치';
        
        html += `
            <div class="category-result ${matchClass}">
                <span class="match-icon">${icon}</span>
                <div class="category-tag">#${item.tag}</div>
                <div class="category-info">카테고리: ${item.category}</div>
                <span class="status-badge ${statusClass}">${statusText}</span>
            </div>
        `;
    });
    
    resultList.innerHTML = html;
    resultArea.classList.add('active');
    
    // 저장
    chrome.storage.local.set({
        lastCheck: {
            mainKeyword: mainKeyword,
            mainCategory: mainCategory,
            results: checkResults,
            timestamp: Date.now()
        }
    });
}

// 내 태그 확인 결과 전송
document.getElementById('btnSendCheckResult').addEventListener('click', async () => {
    const btn = document.getElementById('btnSendCheckResult');
    btn.disabled = true;
    btn.innerHTML = '<span class="loading-spinner"></span>전송 중...';
    
    try {
        const data = await chrome.storage.local.get('lastCheck');
        if (!data.lastCheck) {
            throw new Error('전송할 데이터가 없습니다.');
        }
        
        const response = await fetch(`${SELFMARKETER_API}?action=saveHashtagAnalysis`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                action: 'saveHashtagAnalysis',
                type: 'categoryCheck',
                mainKeyword: data.lastCheck.mainKeyword,
                mainCategory: data.lastCheck.mainCategory,
                results: data.lastCheck.results,
                timestamp: data.lastCheck.timestamp
            })
        });
        
        const result = await response.json();
        
        if (result.success) {
            chrome.tabs.create({
                url: `${SELFMARKETER_API}/mobile-hashtag-analysis.html?view=check&id=${result.id}`
            });
        } else {
            throw new Error(result.error || '전송 실패');
        }
        
    } catch (error) {
        alert('전송 실패: ' + error.message);
    } finally {
        btn.disabled = false;
        btn.innerHTML = '📤 셀프마케터로 전송';
    }
});

// 유틸리티 함수들
function updateProgress(percent, text) {
    document.getElementById('progressFill').style.width = percent + '%';
    document.getElementById('progressText').textContent = text;
}

function updateCheckProgress(percent, text) {
    document.getElementById('checkProgressFill').style.width = percent + '%';
    document.getElementById('checkProgressText').textContent = text;
}

function delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}
